#include "t5disp.h"
#include "mixerhandle.h"
#include <QLabel>
#include <QScrollArea>

t5Disp::t5Disp(snd_hctl_elem_t *elem, QString name, mixerHandle *mixer) {
	m_elem= elem;
	m_mixer= mixer;
	m_name= name;
	snd_ctl_elem_value_malloc(&m_val);
	m_data= new snd_aes_iec958_t;

	m_win= new QWidget();
	m_win->resize(360, 360);
	m_win->setWindowTitle(name + " - IEC958 status");
	m_layout= new QVBoxLayout(m_win);

	m_w0_5= new QGroupBox("Signal format", m_win);
	m_l0_5= new QFormLayout(m_w0_5);
	m_layout->addWidget(m_w0_5);
	m_w6_13= new QGroupBox("Routing", m_win);
	m_l6_13= new QFormLayout(m_w6_13);
	m_layout->addWidget(m_w6_13);
	m_w14_17= new QGroupBox("Local sample address", m_win);
	m_l14_17= new QFormLayout(m_w14_17);
	m_layout->addWidget(m_w14_17);
	m_w18_21= new QGroupBox("Timestamp", m_win);
	m_l18_21= new QFormLayout(m_w18_21);
	m_layout->addWidget(m_w18_21);

	m_layout->addStretch();
	setText("Show IEC958 status...");
}

t5Disp::~t5Disp() {
	snd_ctl_elem_value_free(m_val);
	snd_hctl_elem_set_callback(m_elem, nullptr);
	snd_hctl_elem_set_callback_private(m_elem, nullptr);
}

void t5Disp::activat() {
	snd_hctl_elem_t *elem= snd_hctl_first_elem(m_mixer->m_hctl);
	for(ushort i= 0; i < m_mixer->m_elemCount; i++)
	{
		QString name= snd_hctl_elem_get_name(elem);
		if(name.endsWith("Mask"))
		{
			if(name == "IEC958 " + m_name.left(m_name.indexOf(' ')) + " Con Mask" && snd_hctl_elem_get_index(elem) == snd_hctl_elem_get_index(m_elem))
				m_conmask= elem;
			if(name == "IEC958 " + m_name.left(m_name.indexOf(' ')) + " Pro Mask" && snd_hctl_elem_get_index(elem) == snd_hctl_elem_get_index(m_elem))
				m_promask= elem;
		}
		elem= snd_hctl_elem_next(elem);
	}
	connect(this, SIGNAL(pressed()), this, SLOT(showWin()));
}

void t5Disp::updat() {
	if(m_win->isHidden()) // no need to updat when not opened
		return;

	QLayoutItem *oldItem;
	while((oldItem= m_l0_5->takeAt(0)) != nullptr)
	{
		delete oldItem->widget();
		delete oldItem;
	}
	while((oldItem= m_l6_13->takeAt(0)) != nullptr)
	{
		delete oldItem->widget();
		delete oldItem;
	}
	while((oldItem= m_l14_17->takeAt(0)) != nullptr)
	{
		delete oldItem->widget();
		delete oldItem;
	}
	while((oldItem= m_l18_21->takeAt(0)) != nullptr)
	{
		delete oldItem->widget();
		delete oldItem;
	}
	snd_hctl_elem_read(m_elem, m_val);
	snd_ctl_elem_value_get_iec958(m_val, m_data);

	std::map<uchar, QString> map_emphasis= {
		{ 0, "Not indicated" },
		{ 4, "Disabled" },
		{ 8, "Unknown (0b010)" },
		{ 12, "50/15 μs" },
		{ 16, "Unknown (0b100)" },
		{ 20, "Unknown (0b101)" },
		{ 24, "Unknown (0b110)" },
		{ 28, "CCITT J.17" }
	};
	std::map<uchar, QString> map_samplerate0= {
		{ 64, "48 kHz" },
		{ 128, "44.1 kHz" },
		{ 192, "32 kHz" },
	};
	std::map<uchar, QString> map_chmode= {
		{ 0, "Not indicated" },
		{ 1, "Double freq. (Left channel)" },
		{ 2, "Stereophonic" },
		{ 3, "Unknown (0b0011)" },
		{ 4, "Single channel" },
		{ 5, "Unknown (0b0101)" },
		{ 6, "User-defined (0b0110)" },
		{ 7, "Unknown (0b0111)" },
		{ 8, "2 Ch (Independent)" },
		{ 9, "Double freq. (Right channel)" },
		{ 10, "User-defined (0b1010)" },
		{ 11, "Unknown (0b1011)" },
		{ 12, "Dual language" },
		{ 13, "Unknown (0b1101)" },
		{ 14, "Double freq. (Single channel)" },
		{ 15, "Multichannel" }
	};
	std::map<uchar, QString> map_udatatype= {
		{ 0, "Not indicated" },
		{ 16, "Unknown (0b0001)" },
		{ 32, "General (IEC 60958-3)" },
		{ 48, "Unknown (0b0011)" },
		{ 64, "HDLC" },
		{ 80, "Unknown (0b0101)" },
		{ 96, "Unknown (0b0110)" },
		{ 112, "Unknown (0b0111)" },
		{ 128, "192-bit block" },
		{ 144, "Unknown (0b1001)" },
		{ 160, "Metadata" },
		{ 176, "Unknown (0b1011)" },
		{ 192, "User-defined" },
		{ 208, "Unknown (0b1101)" },
		{ 224, "Unknown (0b1110)" },
		{ 240, "Unknown (0b1111)" }
	};
	std::map<uchar, QString> map_auxsmplbits= {
		{ 0, "Not used (20-bit words)" },
		{ 1, "Unknown (0b001)" },
		{ 2, "Coordination signal" },
		{ 3, "24-bit extension" },
		{ 4, "Unknown (0b100)" },
		{ 5, "Unknown (0b101)" },
		{ 6, "User-defined (0b110)" },
		{ 7, "Unknown (0b111)" },
	};
	std::map<uchar, QString> map_wordlen20= {
		{ 0, "Not indicated" },
		{ 8, "16 bits" },
		{ 16, "18 bits" },
		{ 24, "Unknown (0b011)" },
		{ 32, "19 bits" },
		{ 40, "20 bits" },
		{ 48, "17 bits" },
		{ 56, "Unknown (0b111)" },
	};
	std::map<uchar, QString> map_wordlen24= {
		{ 0, "Not indicated" },
		{ 8, "20 bits" },
		{ 16, "22 bits" },
		{ 24, "Unknown (0b011)" },
		{ 32, "23 bits" },
		{ 40, "24 bits" },
		{ 48, "21 bits" },
		{ 56, "Unknown (0b111)" },
	};
	std::map<uchar, QString> map_aliglvl= {
		{ 0, "Not indicated" },
		{ 64, "EBU R68 (-18.06 dBFS)" },
		{ 128, "SMPTE RP155 (-20 dBFS)" },
		{ 192, "Unknown (0b11)" }
	};
	std::map<uchar, QString> map_dars= {
		{ 0, "No" },
		{ 1, "Grade 2" },
		{ 2, "Grade 1" },
		{ 3, "Unknown (0b11)" }
	};
	std::map<uchar, QString> map_samplerate4= {
		{ 0, "Not indicated" },
		{ 4, "24 kHz" },
		{ 16, "96 kHz" },
		{ 24, "192 kHz" },
		{ 32, "Unknown (0b0100)" },
		{ 40, "Unknown (0b0101)" },
		{ 48, "Unknown (0b0110)" },
		{ 56, "Unknown (0b0111)" },
		{ 64, "Vectored" },
		{ 72, "22.05 kHz" },
		{ 80, "88.2 kHz" },
		{ 88, "176.4 kHz" },
		{ 96, "Unknown (0b1100)" },
		{ 104, "Unknown (0b1101)" },
		{ 112, "Unknown (0b1110)" },
		{ 120, "User-defined" },
	};

	// Status 0
	m_l0_5->addRow("Mode:", new QLabel(m_data->status[0] & 1 ? "Professional (AES3)" : "Consumer (S/PDIF)"));
	snd_ctl_elem_value_t *maskval;
	snd_ctl_elem_value_alloca(&maskval);
	snd_aes_iec958_t *maskdata= new snd_aes_iec958_t;
	if(m_data->status[0] & 1)
		snd_hctl_elem_read(m_promask, maskval);
	else
		snd_hctl_elem_read(m_conmask, maskval);
	snd_ctl_elem_value_get_iec958(maskval, maskdata);

	if(maskdata->status[0] & 2)
		m_l0_5->addRow("Format:", new QLabel(m_data->status[0] & 2 ? "Non-PCM" : "PCM"));
	if(maskdata->status[0] & 0b00011100)
		m_l0_5->addRow("Emphasis:", new QLabel(map_emphasis[m_data->status[0] & 0b00011100]));
	if(maskdata->status[0] & 32)
		m_l0_5->addRow("Sample rate locked:", new QLabel(m_data->status[0] & 32 ? "No" : "Yes"));
	uint rate= 0;
	if(maskdata->status[0] & 0b11000000)
		rate= m_data->status[0] & 0b11000000;
	if(rate)
		m_l0_5->addRow("Sample rate:", new QLabel(map_samplerate0[rate]));
	else if(maskdata->status[4] & 0b01111000)
		m_l0_5->addRow("Sample rate:", new QLabel(map_samplerate4[m_data->status[4] & 0b01111000]));

	// Status 1
	if(maskdata->status[1] & 0b00001111)
		m_l0_5->addRow("Channel mode:", new QLabel(map_chmode[m_data->status[1] & 0b00001111]));
	if(maskdata->status[1] & 0b11110000)
		m_l0_5->addRow("User data type:", new QLabel(map_udatatype[m_data->status[1] & 0b11110000]));

	// Status 2
	if(maskdata->status[2] & 0b00000111)
		m_l0_5->addRow("Aux sample bits:", new QLabel(map_auxsmplbits[m_data->status[2] & 0b00000111]));
	if(maskdata->status[2] & 0b00111000)
	{
		if((m_data->status[2] & 0b00000111) == 3)
			m_l0_5->addRow("Word-length:", new QLabel(map_wordlen24[m_data->status[2] & 0b00111000]));
		else
			m_l0_5->addRow("Word-length:", new QLabel(map_wordlen20[m_data->status[2] & 0b00111000]));
	}
	if(maskdata->status[2] & 0b11000000)
		m_l0_5->addRow("Alignment level:", new QLabel(map_aliglvl[m_data->status[2] & 0b11000000]));

	// Status 3
	if(!maskdata->status[3])
	{
		QString multichmode;
		if(m_data->status[3] < 128)
			multichmode= QString::number(m_data->status[3] + 1) + " ch (undefined)";
		else if((m_data->status[3] & 0b01110000) == 0b01110000)
			multichmode= QString("User-defined (%1 ch)").arg(m_data->status[3] & 0b00001111 + 1);
		else
			multichmode= QString::number((m_data->status[3] & 0b00001111) + 1) + " ch, mode " + QString::number((m_data->status[3] & 0b01110000) / 16);
		m_l0_5->addRow("Multichannel mode:", new QLabel(multichmode));
	}

	// Status 4
	if(maskdata->status[4] & 0b00000011)
		m_l0_5->addRow("AES11 DARS:", new QLabel(map_dars[m_data->status[4] & 0b00000011]));
	if(maskdata->status[4] & 4)
		m_l0_5->addRow("Byte 4 bit 2:", new QLabel(QString::number((m_data->status[4] & 4) / 4)));
	if(maskdata->status[4] & 128)
		m_l0_5->addRow("NTSC adjusted:", new QLabel((m_data->status[4] & 128) ? "Yes" : "No"));

	// Status 5
	if(maskdata->status[5])
		m_l0_5->addRow("Byte 5:", new QLabel(QString("0x%1").arg(m_data->status[5], 2, 16, '0')));

	// Status 6-9
	if(maskdata->status[6] || maskdata->status[9])
	{
		QString chorigin;
		for(uchar i= 9; i > 5; i--)
			chorigin.append(QChar(m_data->status[i]));
		m_l6_13->addRow("Origin:", new QLabel(chorigin));
	}

	// Status 10-13
	if(maskdata->status[10] || maskdata->status[13])
	{
		QString chdest;
		for(uchar i= 13; i > 9; i--)
			chdest.append(QChar(m_data->status[i]));
		if(!chdest.isEmpty())
			m_l6_13->addRow("Destination:", new QLabel(chdest));
	}

	// Status 14-17
	if(maskdata->status[14] || maskdata->status[17])
	{
		QString smpladdr= "0x";
		for(uchar i= 17; i > 13; i--)
			smpladdr.append(QString("%1").arg(m_data->status[i], 2, 16, '0'));
		m_l14_17->addRow("Address:", new QLabel(smpladdr));
	}

	// Status 18-21
	if(maskdata->status[18] || maskdata->status[21])
	{
		QString timestamp= "0x";
		for(uchar i= 21; i > 17; i--)
			timestamp.append(QString("%1").arg(m_data->status[i], 2, 16, '0'));
		m_l18_21->addRow("Timestamp:", new QLabel(timestamp));
	}

	// Status 22
	if(maskdata->status[22] & 16)
		m_l0_5->addRow("Data reliable:", new QLabel(m_data->status[22] & 16 ? "No" : "Yes"));
	if(maskdata->status[22] & 32)
		m_l6_13->addRow("Data reliable:", new QLabel(m_data->status[22] & 32 ? "No" : "Yes"));
	if(maskdata->status[22] & 64)
		m_l14_17->addRow("Data reliable:", new QLabel(m_data->status[22] & 64 ? "No" : "Yes"));
	if(maskdata->status[22] & 128)
		m_l18_21->addRow("Data reliable:", new QLabel(m_data->status[22] & 128 ? "No" : "Yes"));

	delete maskdata;
	if(!m_l0_5->count())
		m_w0_5->hide();
	if(!m_l6_13->count())
		m_w6_13->hide();
	if(!m_l14_17->count())
		m_w14_17->hide();
	if(!m_l18_21->count())
		m_w18_21->hide();
}

void t5Disp::showWin() {
	m_win->show();
	updat();
	snd_hctl_elem_set_callback_private(m_elem, new mixerHandle::callbackData{ mixerHandle::T5, this });
	snd_hctl_elem_set_callback(m_elem, callback);
	m_win->adjustSize();
}
